import streamlit as st
from langchain_openai import OpenAIEmbeddings, ChatOpenAI
from langchain_community.document_loaders import PyPDFLoader
from langchain.text_splitter import RecursiveCharacterTextSplitter
from langchain_community.vectorstores import FAISS
from langchain.prompts import PromptTemplate
from langchain.chains import RetrievalQA
from dotenv import load_dotenv
import tempfile
import os

# Load API key from .env
load_dotenv()

st.set_page_config(page_title="PDF Q&A Chatbot")
st.title("Chat with your PDF (RAG + OpenAI)")


# --- Helper function (cached) ---
@st.cache_resource
def process_pdf(file_path):
    """Process PDF into retriever (runs only once per file)."""
    loader = PyPDFLoader(file_path)
    docs = loader.load()

    splitter = RecursiveCharacterTextSplitter(
        chunk_size=1000,
        chunk_overlap=200
    )
    chunks = splitter.split_documents(docs)

    embeddings = OpenAIEmbeddings(model="text-embedding-3-small")
    vector_store = FAISS.from_documents(chunks, embeddings)

    retriever = vector_store.as_retriever(search_kwargs={"k": 3})
    return retriever


# Upload PDF
uploaded_file = st.file_uploader("Upload a PDF", type=["pdf"])

if uploaded_file:
    if "retriever" not in st.session_state:
        with st.spinner("Processing PDF..."):
            # Save temporarily
            with tempfile.NamedTemporaryFile(delete=False, suffix=".pdf") as tmp_file:
                tmp_file.write(uploaded_file.read())
                tmp_path = tmp_file.name

            # Process once and store retriever
            st.session_state.retriever = process_pdf(tmp_path)

            # Cleanup
            os.remove(tmp_path)

    st.success("PDF processed. Ask questions below:")

    # LLM
    llm = ChatOpenAI(model="gpt-4o-mini", temperature=0.2)

    # Prompt
    prompt = PromptTemplate(
        template="""
        You are a helpful assistant.
        Answer ONLY from the provided context below.
        If the context is insufficient, say you don't know.

        Context:
        {context}

        Question:
        {question}
        """,
        input_variables=["context", "question"]
    )

    qa = RetrievalQA.from_chain_type(
        llm=llm,
        retriever=st.session_state.retriever,
        chain_type="stuff",
        chain_type_kwargs={"prompt": prompt}
    )

    # Chat UI
    user_query = st.text_input("Ask a question about your PDF:")

    if user_query:
        with st.spinner("Thinking..."):
            answer = qa.run(user_query)
        st.markdown(f"**Answer:** {answer}")

else:
    st.info("Please upload a PDF to get started.")
